 /******************************************************************************
  *       __   _   _____   _   _        _____   _____   _____                  *
  *      |  \ | | | ____| | | | |      /  ___| /  ___| /  ___/                 *
  *      |   \| | | |__   | | | |      | |     | |     | |___                  *
  *      | |\   | |  __|  | | | |      | |     | |     \___  \                 *
  *      | | \  | | |___  | |_| |      | |___  | |___   ___| |                 *
  *      |_|  \_| |_____| \_____/      \_____| \_____| /_____/                 *
  *                                                                            *
  *                   RF & Serial Comm. Example                                *
  *                             Receiver                                       *
  ******************************************************************************
  * This program demonstrates the use of the CC2420DB library, including the   *
  * basic RF library and serial link library. The packet protocol being used is*
  * a small subset of the IEEE 802.15.4 standard. It uses an 802.15.4 MAC      *
  * compatible frame format, but does not implement any other MAC functions /  *
  * mechanisms (e.g. CSMA-CA). The basic RF library can thus not be used to    *
  * communicate with compliant 802.15.4 networks.                              *
  *                                                                            *
  * A MSP430F1611 running this program will establish a point-to-point RF link *
  * on channel 26 with a CC2420DB using the following node addresses:          *
  * - PAN ID: 0x2420 (both nodes)                                              *
  * - Short local address: 0x1234                                              *
  * - Transmitter address: 0x5678                                              *
  *                                                                            *
  * The transmitter mote accepts user's input by either press the user-switch  *
  * button on the mote board or a SPACE from the serial link to toggle the blue*
  * LED and transmit the LED value to the receiver mote which will change the  *
  * yellow LED corresponsively.                                                *
  * Please note that there is no so-called (PAN) coordinator.                  *
  *                                                                            *
  * INSTRUCTIONS:                                                              *
  * Incoming data packets contains a 10-byte payload. Byte 0 contains the LED  *
  * value, byte 1 contains the status byte from CC2420.                        *
  * After downloading the program to the device, you may need to press the     *
  * RESET button if it doesn't work.                                           *
  * This example is based on the blinking LED example for the CC2420DB board.  *
  ******************************************************************************
  * Compiler: msp430-gcc GNU C compiler                                        *
  * Target platform: MSP-430F1611 (can easily be ported to other platforms)    *
  ******************************************************************************
  * Last update: 01/16/2006                                                    *
  ******************************************************************************/
		
#include "include/include.h"
#include "include/hardware.h"
//swuart.h is for the software serial communication.
#include "include/swuart.h"
#include <stdlib.h>
#include <stdio.h>



//Defintions used locally in this file.

#define PAYLOAD_SIZE	10
#define RF_CHANNEL	26

volatile static BYTE rf_rx_data= 0;


//Basic RF reception structures.
BASIC_RF_RX_INFO rfRxInfo;
BYTE pRxBuffer[BASIC_RF_MAX_PAYLOAD_SIZE];

/*
 *  BASIC_RF_RX_INFO* basicRfProcessPacket(BASIC_RF_RX_INFO *pRRI)
 *
 *  DESCRIPTION:
 *    This function is a part of the basic RF library, but must be declared by 
 *    the application. Once the application has turned on the receiver, using 
 *    basicRfReceiveOn(), all incoming packets will be received by the FIFOP 
 *    interrupt service routine. When finished, the ISR will call the 
 *    basicRfProcessPacket() function. Please note that this function must 
 *    return quickly, since the next received packet will overwrite the active
 *    BASIC_RF_RX_INFO structure (pointed to by pRRI).
 *
 *  ARGUMENTS:
 *    BASIC_RF_RX_INFO *pRRI
 *      The reception structure, which contains all relevant info about the 
 *      received packet.
 *
 *  RETURN VALUE:
 *    BASIC_RF_RX_INFO*
 *      The pointer to the next BASIC_RF_RX_INFO structure to be used by the 
 *      FIFOP ISR. If there is only one buffer, then return pRRI.
 */
BASIC_RF_RX_INFO* basicRfProcessPacket(BASIC_RF_RX_INFO *pRRI) {
	//Process the received data.	
	rf_rx_data= pRxBuffer[0];
	/* 
	printf("Length(PayLoad): %d\r\n", pRRI->length);
	printf("AckRequest: %x\r\n", pRRI->ackRequest);
	printf("SrcAdd: %x\r\n", pRRI->srcAddr);
	printf("Payload[0]: %x\r\n", pRRI->pPayload[0]);
	printf("Payload[1]: %x\r\n", pRRI->pPayload[1]);
	*/
	return pRRI;
} // basicRfReceivePacket


/*
 *  int main (void)
 *
 *  DESCRIPTION:
 *    Startup routine and main loop
 */
int main (void) {

	//Initalize ports for communication with CC2420 and other peripheral units.
	PORT_INIT();
	SPI_INIT();
	SWUART_INIT();
		
	//Initialize for basic RF operation.
	basicRfInit(&rfRxInfo, RF_CHANNEL, 0x2420, 0x1234);
   
	//Initalize common protocol parameters.
	rfRxInfo.pPayload = pRxBuffer;

	//Turn on RX mode.
	basicRfReceiveOn();

	//The main loop:
	while (1) {
		//Go to low power mode 0.
		LPM0;
		//If any packet received from RF it will exit the low power mode and 
		//execute the following lines.
		printf("Received a packet.\r\n");
		if(rf_rx_data)
			CLR_YLED();
		else
			SET_YLED();
	}
} //main

/*
 *  CC2420 FIFOP interrupt service routine (ISR)
 *
 *  DESCRIPTION:
 *    When a packet has been completely received, this ISR will extract the data
 *    from the RX FIFO, put it into the active BASIC_RF_RX_INFO structure, and 
 *    call basicRfProcessPacket() (defined by the application). FIFO overflow 
 *    and illegally formatted packets are handled by this routine.
 *
 *  Note1: Packets are acknowledged automatically by CC2420 through the auto- 
 *    acknowledgment feature.
 */
interrupt (PORT1_VECTOR) fifo_rx(void){
	//Clearing FIFOP Interrupt.
	CLEAR_FIFOP_INT();
	
	basicRfReceivePacket();
	
	//Wake up the main loop from low power mode 0.
	LPM0_EXIT;
} //fifo_rx
